<?php

namespace App\Http\Controllers\Client;

use App\Models\Product;
use App\Models\Category;
use App\Models\CompanyCategory;
use App\Models\Attribute;
use App\Models\ProductAttribute;
use App\Models\Brand;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = !empty($request->get('q')) ? $request->get('q') : null;
        if($query) {
            $products = Product::Client()->with('category')->search($query, null, true)->paginate(20);
        }else{
            $products = Product::Client()->with('category')->paginate(20);
        }
        return view('clients.products.index', compact('products'));
    }

    public function create()
    {
        $selected_ids = CompanyCategory::where("company_id", auth()->user()->company_id)->pluck('category_id')->all();
        $categories = Category::with('parent')->whereIN('id', $selected_ids)->pluck('name', 'id');
        $brands = Brand::pluck('name', 'id');

        return view('clients.products.create', compact('categories', 'brands'));
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Exception
     */
    public function store(Request $request)
    {
        $product = Product::Client()->create($request->only([
            'category_id',
            'brand_id',
            'name',
            'ref',
            'code_douanier',
            'description',
            'seo_page_title',
            'seo_meta_description',
            'published'
            ]));
        $product->company_id = auth()->user()->company_id;
        $product->slug = str_slug($request->input('name'), '-');
        $product->save();

        if( $request->hasFile('image') ) {
            $file = $request->file('image');
            $path = 'public/companies/'. $product->company_id . '/products/';
            $path = $request->file('image')->storeAs($path, $product->id . '.' . $file->getClientOriginalExtension());
            $product->image = $product->id . '.' . $file->getClientOriginalExtension();
            $product->save();
        }

        if ($request->hasFile('fiche_technique')) {
            $file = $request->file('fiche_technique');
            $path = 'public/companies/'. $product->company_id . '/products/';
            $path = $request->file('fiche_technique')->storeAs($path, 'fiche_technique_' . $product->slug . '.' . $file->getClientOriginalExtension());
            $product->fiche_technique = 'fiche_technique_' . $product->slug . '.' . $file->getClientOriginalExtension();
            $product->save();
        }

        $this->updateAttributes($request, $product);

        return redirect()->route('clients.products.show', $product->id );
    }

    public function show($id)
    {
        $product = Product::Client()->findOrFail($id);
        return view('clients.products.show', compact('product'));
    }

    public function edit($id)
    {
        $selected_ids = CompanyCategory::where("company_id", auth()->user()->company_id)->pluck('category_id')->all();
        $categories = Category::with('parent')->whereIN('id', $selected_ids)->pluck('name', 'id');
        $brands = Brand::pluck('name', 'id');

        $product = Product::Client()->findOrFail($id);

        $attributes = Attribute::where('category_id', $product->category_id)->get();
        $product_attributes = $product->attributes->pluck('value', 'attribute_id');

        return view('clients.products.edit', compact('product', 'categories', 'attributes', 'product_attributes', 'brands'));
    }

    public function update(Request $request, $id)
    {
        $product = Product::Client()->findOrFail($id);
        $product->fill($request->only([
            'category_id',
            'brand_id',
            'name',
            'ref',
            'code_douanier',
            'description',
            'seo_page_title',
            'seo_meta_description',
            'published'
        ]));
        $product->save();

        if( $request->hasFile('image') ) {
            // Delete old file
            $path = 'public/companies/'. $product->company_id . '/products/' . $product->image;
            if(!empty($product->image) && File::exists(storage_path('app/' . $path)) ) {
                Storage::delete($path);
            }
            // Upload new file
            $file = $request->file('image');
            $path = 'public/companies/'. $product->company_id . '/products/';
            $path = $request->file('image')->storeAs($path, $product->id . '.' . $file->getClientOriginalExtension());
            $product->image = $product->id . '.' . $file->getClientOriginalExtension();
            $product->save();
        }

        if ($request->hasFile('fiche_technique')) {
            // Delete old file
            $path = 'public/companies/'. $product->company_id . '/products/' . $product->fiche_technique;
            if (!empty($product->fiche_technique) && File::exists(storage_path('app/' . $path))) {
                Storage::delete($path);
            }
            // Upload new file
            $file = $request->file('fiche_technique');
            $path = 'public/companies/'. $product->company_id . '/products/';
            $path = $request->file('fiche_technique')->storeAs($path, 'fiche_technique_' . $product->slug . '.' . $file->getClientOriginalExtension());
            $product->fiche_technique = 'fiche_technique_' . $product->slug . '.' . $file->getClientOriginalExtension();
            $product->save();
        }

        $this->updateAttributes($request, $product);

        return redirect()->route('clients.products.show', $product->id );
    }

    public function delete($id)
    {
        $product = Product::Client()->find($id);
        $product->delete();

        return redirect()->route('clients.products.index');
    }

    public function attributes($category_id)
    {
        $attributes = Attribute::where('category_id', $category_id)->get();
        return view('clients.products.attributes.index', compact('attributes'));
    }

    /**
     * @param Request $request
     * @param $product
     * @throws \Exception
     */
    public function updateAttributes(Request $request, $product)
    {
        $attributes = array_filter($request->input(), function ($key) {
            return strpos($key, 'attr-') === 0;
        }, ARRAY_FILTER_USE_KEY);

        // Ajout d'attributs
        // Delete All Attributes
        ProductAttribute::where('product_id', $product->id)->delete();
        foreach ($attributes as $key => $attribute) {
            if (strpos($key, "attr-interval-") !== false) {
                $attr_id = str_replace("attr-interval-", "", $key);
                $value = implode(":", $attribute);

            } elseif (strpos($key, "attr-multi-") !== false) {
                $attr_id = str_replace("attr-multi-", "", $key);
                $value = implode("|", $attribute);
            } else {
                $attr_id = str_replace("attr-", "", $key);
                $value = $attribute;
            }

            $productAttribute = ProductAttribute::create([
                'category_id' => $product->category_id,
                'product_id' => $product->id,
                'attribute_id' => $attr_id,
                'value' => $value
            ]);
        }
    }
}
